<?php
class ControllerExtensionThemeFuturehost extends Controller {
    private $error = [];

    /* ========= 基本常數 ========= */
    private const THEME_CODE = 'theme_futurehost';
    private const THEME_DIR  = 'futurehost';

    /* 可選事件（依你的實作路由調整） */
    private const EVENTS = [
        'theme_futurehost_assets' => [
            'trigger' => 'catalog/controller/common/header/before',
            'action'  => 'extension/module/futurehost/asset/index',
        ],
        'theme_futurehost_header' => [
            'trigger' => 'catalog/controller/common/menu/before',
            'action'  => 'extension/module/futurehost/menu/index',
        ],
    ];

    /* ========= 主題預設值 ========= */
    private const DEFAULTS = [
        /* ===============================
        一般設定
        =============================== */
        'directory'                  => self::THEME_DIR,
        'status'                     => 1,
        'product_limit'              => 15,
        'product_description_length' => 100,

        /* ===============================
        圖片尺寸
        =============================== */
        'image_category_width'    => 80,
        'image_category_height'   => 80,
        'image_thumb_width'       => 500,
        'image_thumb_height'      => 500,
        'image_popup_width'       => 800,
        'image_popup_height'      => 800,
        'image_product_width'     => 228,
        'image_product_height'    => 228,
        'image_additional_width'  => 74,
        'image_additional_height' => 74,
        'image_related_width'     => 80,
        'image_related_height'    => 80,
        'image_compare_width'     => 90,
        'image_compare_height'    => 90,
        'image_wishlist_width'    => 47,
        'image_wishlist_height'   => 47,
        'image_cart_width'        => 47,
        'image_cart_height'       => 47,
        'image_location_width'    => 268,
        'image_location_height'   => 50,

        /* ===============================
        主題色系（品牌）
        =============================== */
        'color_primary'   => '#4E7A94',
        'color_secondary' => '#729FDA',
        'color_tertiary'  => '#C2EFEB',
        'color_accent'    => '#FC9464',
        'color_neutral'   => '#b9736e',

        /* ===============================
        狀態色
        =============================== */
        'color_success' => '#28a745',
        'color_warning' => '#ffc107',
        'color_danger'  => '#dc3545',
        'color_info'    => '#17a2b8',
        'color_border'  => '#e6e6e6',

        /* ===============================
        背景與文字
        =============================== */
        'bg_page_default' => '#ffffff',
        'bg_card_default' => '#ffffff',
        'text_default'    => '#1a1a1a',
        'text_secondary'  => '#1a1a1a',
        'text_muted'      => '#6c757d',

        /* ===============================
        按鈕
        =============================== */
        'btn_primary_text'       => '#ffffff',
        'btn_primary_hover_bg'   => '#4E7A94',
        'btn_primary_hover_text' => '#ffffff',
        'btn_secondary_text'       => '#ffffff',
        'btn_secondary_hover_bg'   => '#4E7A94',
        'btn_secondary_hover_text' => '#ffffff',

        /* ===============================
            導覽列與選單
        =============================== */
        'navbar_bg'        => '#383838',
        'navbar_text'      => '#ffffff',
        'menu_bg'          => '#2f2f2f',
        'menu_text'        => '#ffffff',
        'menu_hover_bg'    => '#4E7A94',
        'menu_hover_text'  => '#ffffff',
    ];

    private const INT_KEYS = [
        'status',
        'product_limit',
        'product_description_length',
        'image_category_width','image_category_height',
        'image_thumb_width','image_thumb_height',
        'image_popup_width','image_popup_height',
        'image_product_width','image_product_height',
        'image_additional_width','image_additional_height',
        'image_related_width','image_related_height',
        'image_compare_width','image_compare_height',
        'image_wishlist_width','image_wishlist_height',
        'image_cart_width','image_cart_height',
        'image_location_width','image_location_height',
    ];

    public function index() {
        // 語系 / 標題 / 模型
        $this->load->language('extension/theme/futurehost');
        $this->document->setTitle($this->language->get('heading_title'));
        $this->load->model('setting/setting');

        $user_token = $this->session->data['user_token'] ?? '';
        $store_id   = (int)($this->request->get['store_id'] ?? 0);

        $tabsConfig = [
          'general' => ['name' => $this->language->get('tab_general')],
          'color'   => ['name' => $this->language->get('tab_color')],
          'image'   => ['name' => $this->language->get('tab_image')],
          'product' => ['name' => $this->language->get('tab_product')],
        ];
        $current = $this->request->get['tab'] ?? '';
        if (!isset($tabsConfig[$current])) $current = array_key_first($tabsConfig);

        // 讀 DB 設定（同一個 code 下的所有鍵）
        $setting_info = $this->model_setting_setting->getSetting(self::THEME_CODE, $store_id);

        // 儲存
        if ($this->request->server['REQUEST_METHOD'] === 'POST' && $this->validate()) {
            $to_save = [];
            foreach (self::DEFAULTS as $k => $def) {
                $full = self::k($k);

                // 先用舊值，有 POST 才覆蓋，都沒有才用預設
                $val = array_key_exists($full, $setting_info) ? $setting_info[$full] : $def;
                if (array_key_exists($full, $this->request->post)) {
                    $val = $this->request->post[$full];
                }
                if (in_array($k, self::INT_KEYS, true)) $val = (int)$val;

                $to_save[$full] = $val;
            }

            $this->model_setting_setting->editSetting(self::THEME_CODE, $to_save, $store_id);

            // 指派主題（確保前台抓的是 theme_futurehost_*）
            $this->model_setting_setting->editSettingValue('config', 'config_theme', self::THEME_DIR, $store_id);
            $this->model_setting_setting->editSettingValue(self::THEME_CODE, self::k('directory'), self::THEME_DIR, $store_id);
            $this->model_setting_setting->editSettingValue(self::THEME_CODE, self::k('status'), 1, $store_id);

            if (!empty(self::EVENTS)) $this->rebuildEvents();

            $this->session->data['success'] = $this->language->get('text_success');
            $this->response->redirect($this->url->link(
                'extension/theme/futurehost',
                'user_token=' . $user_token . '&store_id=' . $store_id . '&tab=' . $current,
                true
            ));
            return;
        }
        // if ($this->request->server['REQUEST_METHOD'] === 'POST') {
        //     file_put_contents(DIR_STORAGE . 'logs/futurehost_post.log', "VALIDATE=" . ($this->validate() ? 'TRUE' : 'FALSE') . "\n", FILE_APPEND);
        // }

        /* ====== 視圖資料 ====== */
        $data['heading_title'] = $this->language->get('heading_title');
        $data['text_edit']     = $this->language->get('text_edit');
        $data['text_enabled']  = $this->language->get('text_enabled');
        $data['text_disabled'] = $this->language->get('text_disabled');

        // 訊息/錯誤
        $data['error_warning']                    = $this->error['warning'] ?? '';
        $data['error_product_limit']              = $this->error['product_limit'] ?? '';
        $data['error_product_description_length'] = $this->error['product_description_length'] ?? '';
        $data['error_image_category']             = $this->error['image_category'] ?? '';
        $data['error_image_thumb']                = $this->error['image_thumb'] ?? '';
        $data['error_image_popup']                = $this->error['image_popup'] ?? '';
        $data['error_image_product']              = $this->error['image_product'] ?? '';
        $data['error_image_additional']           = $this->error['image_additional'] ?? '';
        $data['error_image_related']              = $this->error['image_related'] ?? '';
        $data['error_image_compare']              = $this->error['image_compare'] ?? '';
        $data['error_image_wishlist']             = $this->error['image_wishlist'] ?? '';
        $data['error_image_cart']                 = $this->error['image_cart'] ?? '';
        $data['error_image_location']             = $this->error['image_location'] ?? '';
        $data['success']                          = $this->session->data['success'] ?? '';
        unset($this->session->data['success']);

        $data['store_id']  = $store_id;
        $data['user_token']= $user_token;

        // 欄位取值（POST > DB > 預設）
        foreach (self::DEFAULTS as $k => $def) {
            $full = self::k($k);
            $data[$full] = in_array($k, self::INT_KEYS, true)
                ? (int)$this->pv($full, $setting_info, $def)
                : $this->pv($full, $setting_info, $def);
        }

        // 主題目錄清單
        $data['directories'] = [];
        $dirs = glob(DIR_CATALOG . 'view/theme/*', GLOB_ONLYDIR) ?: [];
        foreach ($dirs as $d) {
            $b = basename($d);
            if ($b !== '.' && $b !== '..') $data['directories'][] = $b;
        }
        sort($data['directories'], SORT_NATURAL | SORT_FLAG_CASE);

        // Tabs（連結＋目前分頁）
        $data['tabs'] = array_map(function ($code) use ($tabsConfig, $current, $user_token, $store_id) {
            return [
                'code'   => $code,
                'name'   => $tabsConfig[$code]['name'],
                'href'   => $this->url->link(
                    'extension/theme/futurehost',
                    'user_token=' . $user_token . '&store_id=' . $store_id . '&tab=' . $code,
                    true
                ),
                'active' => ($code === $current),
            ];
        }, array_keys($tabsConfig));

        // 分頁內容（嘗試小寫路由 → 若不存在再試一次大寫 FutureHost）
        $data['tab_content'] = $this->load->view('extension/theme/futurehost/' . $current, $data);

        // Action / Cancel
        $data['action'] = $this->url->link(
            'extension/theme/futurehost',
            'user_token=' . $user_token . '&store_id=' . $store_id . '&tab=' . $current,
            true
        );
        $data['cancel'] = $this->url->link('marketplace/extension', 'user_token=' . $user_token . '&type=theme', true);

        // 標準外框
        $data['header']      = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer']      = $this->load->controller('common/footer');

        $this->response->setOutput($this->load->view('extension/theme/futurehost', $data));
    }

    /* ========= 驗證 ========= */
    protected function validate() {
      // --- 權限檢查 ---
      if (!$this->user->hasPermission('modify', 'extension/theme/futurehost')) {
          $this->error['warning'] = $this->language->get('error_permission');
      }

      // --- 數值欄位檢查 ---
      // 若有填值但不是正數，才記錯誤；空值略過（不阻擋整體儲存）
      $need_pos = [
          'product_limit'              => 'product_limit',
          'product_description_length' => 'product_description_length',
          'image_category_width'   => 'image_category',
          'image_category_height'  => 'image_category',
          'image_thumb_width'      => 'image_thumb',
          'image_thumb_height'     => 'image_thumb',
          'image_popup_width'      => 'image_popup',
          'image_popup_height'     => 'image_popup',
          'image_product_width'    => 'image_product',
          'image_product_height'   => 'image_product',
          'image_additional_width' => 'image_additional',
          'image_additional_height'=> 'image_additional',
          'image_related_width'    => 'image_related',
          'image_related_height'   => 'image_related',
          'image_compare_width'    => 'image_compare',
          'image_compare_height'   => 'image_compare',
          'image_wishlist_width'   => 'image_wishlist',
          'image_wishlist_height'  => 'image_wishlist',
          'image_cart_width'       => 'image_cart',
          'image_cart_height'      => 'image_cart',
          'image_location_width'   => 'image_location',
          'image_location_height'  => 'image_location',
      ];

      foreach ($need_pos as $k => $err) {
          $full = self::k($k);
          $val  = $this->request->post[$full] ?? null;

          // 空值略過，不阻擋儲存
          if ($val === '' || $val === null) continue;

          if (!is_numeric($val) || (int)$val <= 0) {
              $this->error[$err] = $this->language->get(
                  strpos($err, 'image_') === 0 ? 'error_' . $err : 'error_limit'
              );
          }
      }

      // --- 顏色欄位檢查 ---
      // 改成「僅提示錯誤、不阻擋整體儲存」
      $color_keys = [
        // 主題色系
        'color_primary',
        'color_secondary',
        'color_tertiary',
        'color_accent',
        'color_neutral',

        // 狀態提示色
        'color_success',
        'color_warning',
        'color_danger',
        'color_info',
        'color_border',

        // 背景與文字
        'bg_page_default',
        'bg_card_default',
        'text_default',
        'text_secondary',
        'text_muted',

        // 🔘 按鈕
        'btn_primary_text',
        'btn_primary_hover_bg',
        'btn_primary_hover_text',
        'btn_secondary_text',
        'btn_secondary_hover_bg',
        'btn_secondary_hover_text',

        // 導覽列與選單
        'navbar_bg',
        'navbar_text',
        'menu_bg',
        'menu_text',
        'menu_hover_bg',
        'menu_hover_text',
    ];


      // 支援 #FFF 與 #FFFFFF
      $hex = '/^#([0-9a-fA-F]{3}|[0-9a-fA-F]{6})$/';

      $invalid_colors = [];
      foreach ($color_keys as $k) {
          $full = self::k($k);
          $val  = $this->request->post[$full] ?? $this->config->get($full) ?? self::DEFAULTS[$k];

          if ($val && !preg_match($hex, (string)$val)) {
              $invalid_colors[] = $full;
          }
      }

      if (!empty($invalid_colors)) {
          // 不阻擋，只顯示警告訊息
          $this->error['warning'] = ($this->error['warning'] ?? '') .
              ' ' . $this->language->get('error_color_format');
      }

      // --- 最終結果 ---
      // 只要沒有「權限錯誤」或「致命錯誤」，都允許通過
      return empty($this->error['warning']);
    }


    /* ========= 安裝 / 卸載 ========= */
    public function install() {
        $this->load->model('setting/setting');
        $this->load->model('setting/event');
        foreach ($this->getAllStoreIds() as $sid) {
            // 初始化一組預設值
            $init = [];
            foreach (self::DEFAULTS as $k => $v) $init[self::k($k)] = $v;
            $this->model_setting_setting->editSetting(self::THEME_CODE, $init, $sid);

            // 指派主題
            $this->model_setting_setting->editSettingValue('config', 'config_theme', self::THEME_CODE, $sid);
        }
        $this->rebuildEvents();
        $this->clearThemeCache();
    }

    public function uninstall() {
        $this->load->model('setting/setting');
        $this->load->model('setting/event');

        $this->uninstallEvents();

        // 刪主題設定但不強制改回 default（交由你自行切換）
        foreach ($this->getAllStoreIds() as $sid) {
            $this->model_setting_setting->deleteSetting(self::THEME_CODE, $sid);
        }
        $this->clearThemeCache();
    }

    private function rebuildEvents(): void {
        $this->uninstallEvents();
        foreach (self::EVENTS as $code => $def) {
            $this->addOrReplaceEvent($code, $def['trigger'], $def['action']);
        }
    }

    private function uninstallEvents(): void {
        $this->load->model('setting/event');
        foreach (array_keys(self::EVENTS) as $code) {
            $this->model_setting_event->deleteEventByCode($code);
        }
    }

    private function addOrReplaceEvent(string $code, string $trigger, string $action, int $status = 1, int $sort_order = 0): void {
        $this->load->model('setting/event');
        $this->model_setting_event->deleteEventByCode($code);
        $this->model_setting_event->addEvent($code, $trigger, $action, $status, $sort_order);
    }

    /* ========= 小工具 ========= */

    private static function k(string $key): string {
        return self::THEME_CODE . '_' . $key;
    }

    private function pv(string $full_key, array $setting_info, $default) {
        if (isset($this->request->post[$full_key])) return $this->request->post[$full_key];
        if (isset($setting_info[$full_key]))        return $setting_info[$full_key];
        return $default;
    }

    private function getAllStoreIds(): array {
        $ids = [0];
        $this->load->model('setting/store');
        foreach ($this->model_setting_store->getStores() as $s) {
            $ids[] = (int)$s['store_id'];
        }
        return $ids;
    }

    private function clearThemeCache(): void {
        $dir = DIR_STORAGE . 'cache/';
        if (is_dir($dir)) {
            foreach (glob($dir . '*') as $f) if (is_file($f)) @unlink($f);
        }
    }
}
