<?php
class ControllerExtensionModuleFuturehostAsset extends Controller
{
    public function index(): void
    {
        // ---- 1) 判斷目前 route ----
        $route = $this->request->get['route'] ?? 'common/home';

        // ---- 2) 取得正確 base URL（HTTP/HTTPS）----
        $is_https = !empty($this->request->server['HTTPS']) && $this->request->server['HTTPS'] !== 'off';
        $base_url = $is_https ? $this->config->get('config_ssl') : $this->config->get('config_url');

        // ---- 3) 主題名稱（若要固定寫死你自己的主題，可改成 'FutureHost'）----
        $theme = $this->config->get('config_theme') ?: 'futurehost';
        $theme_dir = $this->config->get('theme_futurehost_directory') ?: $theme;

        // ---- 4) 檔案系統實體路徑（用 DIR_APPLICATION 當前台 catalog/ 起點）----
        $theme_fs_base = rtrim(DIR_APPLICATION, '/\\') . '/view/theme/' . $theme_dir . '/';

        // ---- 5) 網址路徑（對應上面的實體目錄）----
        $theme_url_base = rtrim($base_url, '/'). '/catalog/view/theme/' . $theme_dir . '/';

        // ---- 6) 定義各頁面要載入的資產配置 ----
        // 可依需求自行增減，pattern 支援簡單的萬用字元 * 以 starts_with 方式比對
        $map = [
            [
                'pattern' => '*', // 所有頁面都會載入
                'styles'  => [
                    'stylesheet/checkout/cart.css',
                    'stylesheet/common/header.css',
                    'stylesheet/common/footer.css',
                    'stylesheet/extension/module/fh_category_product.css',
                    'stylesheet/extension/module/fh_accordion_builder.css',
                    'stylesheet/extension/module/slideshow.css',
                    'stylesheet/common/success.css',
                    'stylesheet/common/cart.css',
                    // 'stylesheet/stylesheet.css', // 全域主樣式（如果有）
                ],
                'scripts' => [
                    // 'javascript/futurehost/common.js', // 可放共用 JS
                ],
            ],
            [
                'pattern' => 'common/home',
                'styles'  => [
                    'stylesheet/common/home.css',
                ],
                'scripts' => [
                    // 'javascript/futurehost/home.js',
                ],
            ],
            [
                'pattern' => 'product/product',
                'styles'  => [
                    // 'stylesheet/futurehost/base.css',
                    'stylesheet/futurehost/product.css',
                ],
                'scripts' => [
                    // 'javascript/futurehost/product.js',
                ],
            ],
            [
                'pattern' => 'account/login',
                'styles'  => [
                    'stylesheet/account/login.css',
                ],
                'scripts' => [],
            ],
            [
                'pattern' => 'account/register',
                'styles'  => [
                    'stylesheet/account/register.css',
                ],
                'scripts' => [],
            ],
            [
                'pattern' => 'information/contact',
                'styles'  => [
                    'stylesheet/information/contact.css',
                ],
                'scripts' => [],
            ],
            [
                'pattern' => 'product/category',
                'styles'  => [
                    'stylesheet/product/category.css',
                ],
                'scripts' => [],
            ],
            [
                'pattern' => 'product/product',
                'styles'  => [
                    'stylesheet/product/product.css',
                ],
                'scripts' => [
                    'javascript/product.js',
                ],
            ],
        ];

        // ---- 7) 依照 route 套用資產 ----
        //    會將符合的多組 pattern 合併（先專頁、最後 * fallback）
        $resolved = [
            'styles'  => [],
            'scripts' => [],
        ];

        foreach ($map as $rule) {
            if ($this->routeMatch($route, $rule['pattern'])) {
                $resolved['styles']  = array_merge($resolved['styles'],  $rule['styles']);
                $resolved['scripts'] = array_merge($resolved['scripts'], $rule['scripts']);
            }
        }

        // ---- 8) 實際注入資產 ----
        foreach (array_unique($resolved['styles']) as $rel_path) {
            $this->safeAddStyle($theme_fs_base, $theme_url_base, $rel_path);
        }
        foreach (array_unique($resolved['scripts']) as $rel_path) {
            $this->safeAddScript($theme_fs_base, $theme_url_base, $rel_path);
        }

        // ---- 9) 顏色變數注入 ----
        $this->injectThemeColors();
    }

    private function injectThemeColors(): void
    {
      // --- 1️⃣ 取得主題顏色設定（資料庫或預設值） ---
    $colors = [
        'primary'   => $this->config->get('theme_futurehost_color_primary')   ?: '#4E7A94',
        'secondary' => $this->config->get('theme_futurehost_color_secondary') ?: '#729FDA',
        'tertiary'  => $this->config->get('theme_futurehost_color_tertiary')  ?: '#C2EFEB',
        'accent'    => $this->config->get('theme_futurehost_color_accent')    ?: '#FC9464',
        'neutral'   => $this->config->get('theme_futurehost_color_neutral')   ?: '#ECFEE8',
    
        'success'   => $this->config->get('theme_futurehost_color_success')   ?: '#28a745',
        'warning'   => $this->config->get('theme_futurehost_color_warning')   ?: '#ffc107',
        'danger'    => $this->config->get('theme_futurehost_color_danger')    ?: '#dc3545',
        'info'      => $this->config->get('theme_futurehost_color_info')      ?: '#17a2b8',
        'border'    => $this->config->get('theme_futurehost_color_border')    ?: '#e6e6e6',
    
        'bg_page_default' => $this->config->get('theme_futurehost_bg_page_default') ?: '#ffffff',
        'bg_card_default' => $this->config->get('theme_futurehost_bg_card_default') ?: '#ffffff',
        'text_default'    => $this->config->get('theme_futurehost_text_default')    ?: '#1a1a1a',
        'text_secondary'  => $this->config->get('theme_futurehost_text_secondary')  ?: '#1a1a1a',
        'text_muted'      => $this->config->get('theme_futurehost_text_muted')      ?: '#6c757d',

        'btn_primary_text'       => $this->config->get('theme_futurehost_btn_primary_text')       ?: '#ffffff',
        'btn_primary_hover_bg'   => $this->config->get('theme_futurehost_btn_primary_hover_bg')   ?: '#4E7A94',
        'btn_primary_hover_text' => $this->config->get('theme_futurehost_btn_primary_hover_text') ?: '#ffffff',
        'btn_secondary_text'       => $this->config->get('theme_futurehost_btn_secondary_text')       ?: '#ffffff',
        'btn_secondary_hover_bg'   => $this->config->get('theme_futurehost_btn_secondary_hover_bg')   ?: '#4E7A94',
        'btn_secondary_hover_text' => $this->config->get('theme_futurehost_btn_secondary_hover_text') ?: '#ffffff',
    
        'navbar_bg'        => $this->config->get('theme_futurehost_navbar_bg')        ?: '#383838',
        'navbar_text'      => $this->config->get('theme_futurehost_navbar_text')      ?: '#ffffff',
        'menu_bg'          => $this->config->get('theme_futurehost_menu_bg')          ?: '#2f2f2f',
        'menu_text'        => $this->config->get('theme_futurehost_menu_text')        ?: '#ffffff',
        'menu_hover_bg'    => $this->config->get('theme_futurehost_menu_hover_bg')    ?: '#4E7A94',
        'menu_hover_text'  => $this->config->get('theme_futurehost_menu_hover_text')  ?: '#ffffff',
    ];

      // --- 2️⃣ 組出完整 CSS ---
$css = <<<CSS
:root {
--color-primary:   {$colors['primary']};
--color-secondary: {$colors['secondary']};
--color-tertiary:  {$colors['tertiary']};
--color-accent:    {$colors['accent']};
--color-neutral:   {$colors['neutral']};

--color-success: {$colors['success']};
--color-warning: {$colors['warning']};
--color-danger:  {$colors['danger']};
--color-info:    {$colors['info']};
--color-border:  {$colors['border']};

--bg-page-default: {$colors['bg_page_default']};
--bg-card-default: {$colors['bg_card_default']};
--text-default:    {$colors['text_default']};
--text-secondary:  {$colors['text_secondary']};
--text-muted:      {$colors['text_muted']};

--btn-primary-bg:         {$colors['primary']};
--btn-primary-text:       {$colors['btn_primary_text']};
--btn-primary-hover-bg:   {$colors['btn_primary_hover_bg']};
--btn-primary-hover-text: {$colors['btn_primary_hover_text']};
--btn-secondary-bg:         {$colors['secondary']};
--btn-secondary-text:       {$colors['btn_secondary_text']};
--btn-secondary-hover-bg:   {$colors['btn_secondary_hover_bg']};
--btn-secondary-hover-text: {$colors['btn_secondary_hover_text']};

--navbar-bg:        {$colors['navbar_bg']};
--navbar-text:      {$colors['navbar_text']};
--menu-bg:          {$colors['menu_bg']};
--menu-text:        {$colors['menu_text']};
--menu-hover-bg:    {$colors['menu_hover_bg']};
--menu-hover-text:  {$colors['menu_hover_text']};
}
CSS;

      // --- 3️⃣ 確保暫存目錄存在 ---
      $temp_dir = rtrim(DIR_APPLICATION, '/\\') . '/view/theme/futurehost/temp/';
      if (!is_dir($temp_dir)) {
          if (!mkdir($temp_dir, 0755, true) && !is_dir($temp_dir)) {
              error_log('[FutureHost] 無法建立顏色暫存資料夾：' . $temp_dir);
              return;
          }
      }

      // --- 4️⃣ 寫入 CSS 檔案（僅當內容變更時重寫） ---
      $temp_file = $temp_dir . 'theme-colors.css';
      $hash_file = $temp_dir . 'theme-colors.hash';
      $new_hash  = md5($css);
      $old_hash  = is_file($hash_file) ? file_get_contents($hash_file) : '';

      if ($new_hash !== $old_hash) {
          file_put_contents($temp_file, $css);
          file_put_contents($hash_file, $new_hash);
      }

      // --- 5️⃣ 產出對應 URL ---
      $is_https = !empty($this->request->server['HTTPS']) && $this->request->server['HTTPS'] !== 'off';
      $base_url = $is_https
          ? rtrim($this->config->get('config_ssl'), '/')
          : rtrim($this->config->get('config_url'), '/');

      $theme_css_url = $base_url . '/catalog/view/theme/futurehost/temp/theme-colors.css?v=' . filemtime($temp_file);

      // --- 6️⃣ 注入 <head> ---
      $this->document->addStyle($theme_css_url);
    }

    private function routeMatch(string $route, string $pattern): bool
    {
        if ($pattern === '*') {
            return true;
        }
        // 簡單的「以 pattern 去掉 * 後的前綴」比對
        if (substr($pattern, -1) === '*') {
            $prefix = rtrim($pattern, '*');
            return strpos($route, $prefix) === 0;
        }
        return $route === $pattern;
    }

    private function safeAddStyle(string $fs_base, string $url_base, string $relative): void
    {
        $fs = $fs_base . ltrim($relative, '/');
        $url = $url_base . ltrim($relative, '/');

        if (is_file($fs)) {
            $ver = (string) @filemtime($fs);
            $this->document->addStyle($url . '?v=' . $ver);
        } else {
            error_log("[FutureHost Asset] CSS not found: {$fs}");
        }
    }

    private function safeAddScript(string $fs_base, string $url_base, string $relative): void
    {
        $fs = $fs_base . ltrim($relative, '/');
        $url = $url_base . ltrim($relative, '/');

        if (is_file($fs)) {
            $ver = (string) @filemtime($fs);
            $this->document->addScript($url . '?v=' . $ver);
        } else {
            error_log("[FutureHost Asset] JS not found: {$fs}");
        }
    }
}
